package tests::OutputFormatsHTMLTest;

use strict;

use base qw/Lire::Test::PluginTestCase/;

use Lire::OutputFormats::HTML;
use Lire::PluginManager;
use Lire::Utils qw/tempdir create_file/;
use File::Basename qw/dirname/;
use File::Path qw/rmtree /;
use Lire::Test::Mock;

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    $self->{'testdir'} = dirname( __FILE__ ) ;
    $self->{'tmpdir'} = tempdir( $self->name() . '_XXXXXX' );
    return;
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();

    Lire::Test::Mock->reset_factories();
    rmtree( $self->{'tmpdir'} );

    return;
}

sub create_plugin {
    return new Lire::OutputFormats::HTML();
}

sub properties_spec_file {
    return dirname( __FILE__ ) . "/../Lire/OutputFormats/of_html.xml";
}

sub registration_file {
    return dirname( __FILE__ ) . "/../Lire/OutputFormats/of_html_init";
}

sub test_format_report {
    my $self = $_[0];

    $self->test_registration_file();
    Lire::Test::Mock->set_mock_factory( 'Lire::ReportParser::HTMLWriter',
                                        'write_report' => '' );
    my $html = Lire::PluginManager->get_plugin( 'output_format', 'html' );
    $html->format_report( "$self->{'testdir'}/data/test-sqlite.xml",
                          $self->{'tmpdir'},
                          { 'one_page' => 0, 'xhtml' => 0 } );
    my ($writer) = @{Lire::Test::Mock->mock_instances( 'Lire::ReportParser::HTMLWriter' )};
    my ( $ob, $report, $dir, %args ) = 
      @{$writer->get_invocation( 'write_report' )};
    $self->assert_isa( 'Lire::Report', $report );
    $self->assert_str_equals( $self->{'tmpdir'}, $dir );
    $self->assert_num_equals( 0, $args{'xhtml'} );
    $self->assert_num_equals( 0, $args{'one_page'} );
}

sub test_mime_report {
   my $self = $_[0];

   my $page = <<EOPAGE;
<a href="http://www.logreport.org/">A link</href>
<link href="lire.css" />
<img src="chart1.jpg" />
<img src="chart2.jpg" />
<img src="http://www.logreport.org/lire.png" />
EOPAGE
   $self->test_registration_file();
   my %args = ();
   Lire::Test::Mock->set_mock_factory( 'Lire::ReportParser::HTMLWriter',
                                       'write_report' => 
                                       sub { my $dir = $_[2];
                                             %args = @_[3..$#_];
                                             create_file( "$dir/index.html", $page );
                                             create_file( "$dir/chart1.jpg" );
                                             create_file( "$dir/chart2.png" );
                                             create_file( "$dir/lire.css" );
                                             create_file( "$dir/lire.png" );
                                         } );
   my $html = Lire::PluginManager->get_plugin( 'output_format', 'html' );

   my $entity = $html->mime_report( "$self->{'testdir'}/data/test-sqlite.xml" );
   $self->assert_num_equals( 1, $args{'one_page'} );
   $self->assert_isa( 'MIME::Entity', $entity);
   $self->assert_str_equals( 'multipart/related',
                             $entity->head()->mime_type() );
   $self->assert_num_equals( 5, scalar $entity->parts() );
   my $part = $entity->parts(0);
   $self->assert_str_equals( 'text/html', $part->head()->mime_type() );
   $self->assert_str_equals( "quoted-printable\n",
                             $part->head()->get( 'content-transfer-encoding' ) );
   $self->assert_str_equals( 'utf-8', $part->head()->mime_attr( 'content-type.charset' ) );
   $self->assert_str_equals( <<EOF, $part->bodyhandle()->as_string() );
<a href="http://www.logreport.org/">A link</href>
<link href="cid:lire.css" />
<img src="cid:chart1.jpg" />
<img src="cid:chart2.jpg" />
<img src="http://www.logreport.org/lire.png" />
EOF
   $part = $entity->parts(1);
   $self->assert_str_equals( 'image/jpeg', $part->head()->mime_type() );
   $self->assert_str_equals( "<chart1.jpg>\n", $part->head()->get( 'content-id' ) );
   $self->assert_str_equals( "base64\n", 
                             $part->head()->get( 'content-transfer-encoding' ) );

   $part = $entity->parts(2);
   $self->assert_str_equals( 'image/png', $part->head()->mime_type() );
   $self->assert_str_equals( "<chart2.png>\n", $part->head()->get( 'content-id' ) );
   $self->assert_str_equals( "base64\n", 
                             $part->head()->get( 'content-transfer-encoding' ) );
   $part = $entity->parts(3);
   $self->assert_str_equals( 'text/css', $part->head()->mime_type() );
   $self->assert_str_equals( "<lire.css>\n", $part->head()->get( 'content-id' ) );
   $self->assert_str_equals( "quoted-printable\n", 
                             $part->head()->get( 'content-transfer-encoding' ) );
}

1;
